
#ifndef INTER_PROCESSOR_REQUEST_H
#define INTER_PROCESSOR_REQUEST_H

#include <dtvrecdd/memtrans.h>
#include "iprq_cfg.h"

#define IPRQ_ABORT()		\
do {				\
	while(1);		\
} while(0)


#define MEMBLK_MIN_SHIFT	7	/* 128B */
//#define MEMBLK_MIN_SHIFT	12	/* 4KB */
#define MEMBLK_MAX_SHIFT	18	/* 256KB */

#define MEMBLK_MIN_SIZE		(1 << MEMBLK_MIN_SHIFT)
#define MEMBLK_MAX_SIZE		(1 << MEMBLK_MAX_SHIFT)
#define MEMBLK_MIN_MASK		(~(MEMBLK_MIN_SIZE - 1))
#define MEMBLK_MAX_MASK		(~(MEMBLK_MAX_SIZE - 1))

#define MEMBLK_ALIGN_SHIFT	12	/* 4KB == PAGE_SHIFT */
#define MEMBLK_ALIGN_SIZE	(1 << MEMBLK_ALIGN_SHIFT)
#define MEMBLK_ALIGN_MASK	(~(MEMBLK_ALIGN_SIZE - 1))

#define MEMBLK_NUM(size)	(((size) + MEMBLK_MIN_SIZE - 1) >> MEMBLK_MIN_SHIFT)

#define IPRQ_MEM_PADDR		((CONFIG_IPRQ_MEM_PADDR + MEMBLK_ALIGN_SIZE - 1) & MEMBLK_ALIGN_MASK)
#define IPRQ_MEM_SIZE		((CONFIG_IPRQ_MEM_PADDR + CONFIG_IPRQ_MEM_SIZE - IPRQ_MEM_PADDR) & MEMBLK_ALIGN_MASK)

#define MEMBLK_ALL_NUM		MEMBLK_NUM(IPRQ_MEM_SIZE)

#define IPRQ_MEM_ADDR iprq_mem_addr
extern unsigned long iprq_mem_addr;

/* memblk define check */
/* only max size overflow should be checked - IPRQ_MEM_SIZE varies by configure */
#if (MEMBLK_MAX_SIZE > IPRQ_MEM_SIZE) /* || (MEMBLK_MAX_SIZE <= (IPRQ_MEM_SIZE / 2)) */
#error MEMBLK_MAX_SHIFT incorrectly.
#endif

#define __PREQ_ASYNC	0
#define __PREQ_SYNC	1

struct proc_request_t {
	int request;
	int sender;
	int sync;
	unsigned int size;
	unsigned int arg;
};

struct iprq_ctlblk_t {
	volatile int lock;
	int retval;
	int finish;

	int remain;
	int first;
	int last;
	struct proc_request_t buf[IPRQ_REQUEST_BUFSIZE];
};

typedef int (*iprq_handler_t)(int, void *);

struct iprq_memblk_t {
	volatile int lock;
	unsigned char flag[MEMBLK_ALL_NUM];
	unsigned int unused_num[32];
};

struct iprq_printcb_t {
	volatile int read_lock;
	volatile int write_lock;
	int read_offset;
	int write_offset;
	char buf[IPRQ_PRINT_BUFSIZE];
};

struct iprq_status_t {
	volatile int lock;
	unsigned int initialized_os_mask;
};

#define MEMBLK_FLAG_SIZE_MASK	0x1f
#define MEMBLK_FLAG_USED	0x40
#define MEMBLK_FLAG_RESERVED	0x80

#define memblk_start_addr(memblk)		IPRQ_MEM_ADDR

#define iprq_memid_to_index(id)			((id) >> MEMBLK_MIN_SHIFT)
#define iprq_index_to_memid(index)		((index) << MEMBLK_MIN_SHIFT)
#define iprq_memid_to_offset(id)		(id)
#define iprq_offset_to_memid(offset)		(offset)
#define iprq_memid_to_addr(memblk, id)		(memblk_start_addr(memblk) + iprq_memid_to_offset(id))
#define iprq_addr_to_memid(memblk, addr)	iprq_offset_to_memid((unsigned long)(addr) - memblk_start_addr(memblk))

#define iprq_virt_to_phys(addr)			((unsigned long)(addr) - IPRQ_MEM_ADDR + IPRQ_MEM_PADDR)
#define iprq_phys_to_virt(addr)			((unsigned long)(addr) - IPRQ_MEM_PADDR + IPRQ_MEM_ADDR)

#define IPRQ_MEMBLK_START		(IPRQ_MEM_ADDR + IPRQ_MEM_SIZE - sizeof(struct iprq_memblk_t))
#define IPRQ_CTLBLK_START		(IPRQ_MEMBLK_START - sizeof(struct iprq_ctlblk_t) * OS_NUM)
#define IPRQ_PRINTCB_START		(IPRQ_CTLBLK_START - sizeof(struct iprq_printcb_t))
#define IPRQ_STATUS_START		(IPRQ_PRINTCB_START - sizeof(struct iprq_status_t))

#define iprq_ctlblk			((struct iprq_ctlblk_t *)IPRQ_CTLBLK_START)
#define iprq_memblk			((struct iprq_memblk_t *)IPRQ_MEMBLK_START)
#define iprq_printcb			((struct iprq_printcb_t *)IPRQ_PRINTCB_START)
#define iprq_status			((struct iprq_status_t *)IPRQ_STATUS_START)

#define MEMBLK_SYSTEM_SIZE		(sizeof(struct iprq_memblk_t) + (sizeof(struct iprq_ctlblk_t) * OS_NUM) + sizeof (struct iprq_printcb_t) + sizeof (struct iprq_status_t))

/* inter processor shared memory */
static inline int __is_memblk_used(struct iprq_memblk_t *memblk, int index)
{
	if (memblk->flag[index] & MEMBLK_FLAG_USED) {
		return 1;
	}
	return 0;
}

static inline int __is_memblk_reserved(struct iprq_memblk_t *memblk, int index)
{
	if (memblk->flag[index] & MEMBLK_FLAG_RESERVED) {
		return 1;
	}
	return 0;
}

static inline void __memblk_set_used(struct iprq_memblk_t *memblk, int index)
{
	memblk->flag[index] |= MEMBLK_FLAG_USED;
}

static inline void __memblk_clear_used(struct iprq_memblk_t *memblk, int index)
{
	memblk->flag[index] &= ~MEMBLK_FLAG_USED;
}

static inline int __memblk_get_block_size(struct iprq_memblk_t *memblk, int index)
{
	return memblk->flag[index] & MEMBLK_FLAG_SIZE_MASK;
}

static inline void __memblk_set_block_size(struct iprq_memblk_t *memblk, int index, int shift)
{
	memblk->flag[index] &= ~MEMBLK_FLAG_SIZE_MASK;
	memblk->flag[index] |= (shift & MEMBLK_FLAG_SIZE_MASK);
}

static inline void __memblk_set_reserved(struct iprq_memblk_t *memblk, int index)
{
	memblk->flag[index] |= MEMBLK_FLAG_RESERVED;
}

static inline void __init_iprq_mem(void)
{
	struct iprq_memblk_t *memblk = iprq_memblk;
	int sysstart = MEMBLK_ALL_NUM - MEMBLK_NUM(MEMBLK_SYSTEM_SIZE);
	int i, shift, start, len;

	memblk->lock = 0;

	for (i = 0; i < 32; i++) {
		memblk->unused_num[i] = 0;
	}
	for (i = 0; i < MEMBLK_ALL_NUM; i++) {
		memblk->flag[i] = 0;
	}

	start = 0;
	for (shift = MEMBLK_MAX_SHIFT; shift >= MEMBLK_MIN_SHIFT; shift--) {
		len = 1 << (shift - MEMBLK_MIN_SHIFT);
		if (start + len > sysstart) {
			continue;
		}
		for (i = 0; i < len; i++) {
			__memblk_set_block_size(memblk, start + i, shift);
		}
		memblk->unused_num[shift]++;
		start += len;
	}
	for (i = sysstart; i < MEMBLK_ALL_NUM; i++) {
		__memblk_set_reserved(memblk, i);
		__memblk_set_used(memblk, i);
	}
}

static inline int __is_os_initialized(struct iprq_status_t *status, int os)
{
	if (status->initialized_os_mask & (1 << (os))) {
		return 1;
	}
	return 0;
}

static inline int __is_all_os_initialized(struct iprq_status_t *status)
{
	if ((status->initialized_os_mask ^ ((1 << OS_NUM) - 1)) == 0) {
		return 1;
	}
	return 0;
}

static inline void __set_os_initialized(struct iprq_status_t *status, int os)
{
	status->initialized_os_mask |= (1 << (os));
}

static inline void __init_iprq_status(void)
{
	struct iprq_status_t *status = iprq_status;

	status->lock = 0;
	status->initialized_os_mask = 0;
}

/* inter processor request */
static inline void __init_proc_request(int osid)
{
	struct iprq_ctlblk_t *ctlblk = &iprq_ctlblk[osid];
	ctlblk->first = ctlblk->last = 0;
	ctlblk->remain = 0;
	ctlblk->lock = 0;
}

extern int __iprq_initialized;
#define is_iprq_initialized()	__iprq_initialized

extern void do_proc_request(void);
extern void send_iprq(int osid);

extern int iprq_is_initialized(void);

/* iprq API */
extern int iprqX_send(int osid, int req, void *msg, unsigned int size);
extern int iprqX_send_sync(int osid, int req, void *msg, unsigned int size, int *ret);
extern int iprqX_send_sync_timeout(int osid, int req, void *msg, unsigned int size, int *ret, unsigned int usec);
extern int iprqX_entry(int req, iprq_handler_t handler);

extern int iprqX_alloc_mem(unsigned int size);
extern int iprqX_free_mem(int memid);
extern void *iprqX_attach_mem(int memid);
extern int iprqX_detach_mem(void *ptr);

/* inter processor request */
#define IPRQ_WAKE_UP		0x00
#define IPRQ_SEND_PACKET	0x01
#define IPRQ_MALLOC		0x02
#define IPRQ_FREE		0x03
#define IPRQ_BLOCK_REQUEST	0x04
#define IPRQ_GET_STATE		0x05
#define IPRQ_LINUX_BOOTED	0x06


#define LISTID(major, minor, queue, packet, data)	\
	((data) << 24 | (packet) << 20 | (queue) << 16 | ((major) & 0xff) << 8 | ((minor) & 0xff))

#define GET_MAJOR(listid)	(((unsigned int)(listid) >> 8) & 0xff)
#define GET_MINOR(listid)	((unsigned int)(listid) & 0xff)
#define GET_QUE_TYPE(listid)	(((unsigned int)(listid) >> 16) & 0xf)
#define GET_PACKET_TYPE(listid)	(((unsigned int)(listid) >> 20) & 0xf)
#define GET_DATA(listid)	(((unsigned int)(listid) >> 24) & 0xff)

/* listid queue type */
#define LID_QUE_OPEN		0
#define LID_QUE_REQUEST		1
#define LID_QUE_REPLY		2
#define LID_QUE_BLOCK		3
#define LID_QUE_FIFO		4

/* listid packet type */
#define LID_PACKET_NONE		0
#define LID_PACKET_DAT		1
#define LID_PACKET_CTL		2

/* IPRQ_GET_STATE arg */
#define STATE_OPEN_FLAG		0
#define STATE_TYPE		1


/* IPRQ error */
#define E_IPRQ_INVAL		-1
#define E_IPRQ_BUSY		-2
#define E_IPRQ_TIMEOUT		-3
#define E_IPRQ_NOMEM		-4


#define iprqL_send(osid, req, msg, size)		\
	iprqX_send(osid, req, msg, size)
#define iprqL_send_sync(osid, req, msg, size, ret)	\
	iprqX_send_sync(osid, req, msg, size, ret)
#define iprqL_send_sync_timeout(osid, req, msg, size, ret, usec)	\
	iprqX_send_sync_timeout(osid, req, msg, size, ret, usec)
#define iprqL_entry(req, handler)			\
	iprqX_entry(req, (iprq_handler_t)(handler))

static inline void LLIST_ADD_TAIL(int memid, unsigned int listid)
{
	unsigned int msg[2] = {listid, (unsigned int)memid};
	while(iprqL_send(OS_R, IPRQ_SEND_PACKET, msg, 8));
}

static inline void LWAKE_UP_INTERRUPTIBLE(unsigned int listid)
{
	while(iprqL_send_sync(OS_R, IPRQ_WAKE_UP, &listid, 4, NULL));
}

static inline void LBLOCK_REQUEST(unsigned int listid, int memid)
{
	unsigned int msg[2] = {listid, (unsigned int)memid};
	while(iprqL_send_sync(OS_R, IPRQ_BLOCK_REQUEST, msg, 8, NULL));
}

static inline int LGET_BUDDY_OPEN_FLAG(int major, int minor)
{
	int retval;
	unsigned int msg[2] = {STATE_OPEN_FLAG, (((unsigned int)major & 0xff) << 8) | ((unsigned int)minor & 0xff)};
	while(iprqL_send_sync(OS_R, IPRQ_GET_STATE, msg, 8, &retval));
	return retval;
}

static inline int LGET_BUDDY_TYPE(int major, int minor)
{
	int retval;
	unsigned int msg[2] = {STATE_TYPE, (((unsigned int)major & 0xff) << 8) | ((unsigned int)minor & 0xff)};
	while(iprqL_send_sync(OS_R, IPRQ_GET_STATE, msg, 8, &retval));
	return retval;
}

#define iprqL_alloc_mem(size)	iprqX_alloc_mem(size)
#define iprqL_free_mem(id)	iprqX_free_mem(id)
#define iprqL_attach_mem(id)	iprqX_attach_mem(id)
#define iprqL_detach_mem(ptr)	iprqX_detach_mem(ptr)

#define LALLOC_MEM(size)	iprqL_alloc_mem(size)
#define LFREE_MEM(id)		iprqL_free_mem(id)
#define LATTACH_MEM(id)		iprqL_attach_mem(id)
#define LDETACH_MEM(ptr)	iprqL_detach_mem(ptr)

static inline void* LATTACH_MEM_PADDR(int id)
{
	void* paddr = LATTACH_MEM(id);
	if (!paddr) {
		return NULL;
	}
	return (void*)iprq_virt_to_phys(paddr);
}

static inline int LDETACH_MEM_PADDR(void* ptr)
{
	return 	LDETACH_MEM((void*)iprq_phys_to_virt(ptr));
}

#define LFREE_MEM_ADDR(ptr)				\
do {							\
	int id;						\
	id = iprq_addr_to_memid(iprq_memblk, ptr);	\
	iprqL_free_mem(id);				\
} while (0)

#endif /* INTER_PROCESSOR_REQUEST_H */
